<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Handle mark all as read action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'mark_all_read') {
    try {
        // Start transaction
        $pdo->beginTransaction();
        
        // Mark notifications as read
        $stmt = $pdo->prepare("UPDATE notifications SET is_read = 1 WHERE is_read = 0");
        $stmt->execute();
        
        // Mark overdue credits as read by updating their status
        $stmt = $pdo->prepare("
            UPDATE credit_transactions ct
            JOIN customers c ON ct.customer_id = c.id
            SET ct.status = 'read'
            WHERE ct.type = 'credit' 
            AND ct.status = 'pending'
            AND EXISTS (
                SELECT 1 
                FROM (
                    SELECT customer_id,
                           SUM(CASE WHEN type = 'credit' THEN amount ELSE 0 END) - 
                           SUM(CASE WHEN type = 'payment' THEN amount ELSE 0 END) as balance
                    FROM credit_transactions
                    GROUP BY customer_id
                    HAVING balance > 0
                ) as unpaid
                WHERE unpaid.customer_id = ct.customer_id
            )
        ");
        $stmt->execute();
        
        // Commit transaction
        $pdo->commit();
        
        // If it's an AJAX request, return success
        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
            echo json_encode(['success' => true]);
            exit;
        }
    } catch (PDOException $e) {
        // Rollback transaction on error
        $pdo->rollBack();
        $error = "Error marking notifications as read: " . $e->getMessage();
        
        // If it's an AJAX request, return error
        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
            echo json_encode(['success' => false, 'error' => $error]);
            exit;
        }
    }
}

// Get email and SMS settings
try {
    $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group IN ('email_settings', 'sms_settings')");
    $stmt->execute();
    $apiSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
} catch (PDOException $e) {
    $error = "Error fetching API settings: " . $e->getMessage();
}

// Get overdue creditors
try {
    $stmt = $pdo->prepare("
        SELECT 
            c.id, 
            c.name,
            c.email,
            c.phone,
            c.credit_limit,
            MAX(ct.created_at) as last_payment_date,
            COALESCE(SUM(CASE WHEN ct.type = 'credit' THEN ct.amount ELSE 0 END), 0) as total_credit_used,
            COALESCE(SUM(CASE WHEN ct.type = 'payment' THEN ct.amount ELSE 0 END), 0) as total_payments,
            DATEDIFF(CURRENT_DATE, MAX(ct.created_at)) as days_overdue,
            (COALESCE(SUM(CASE WHEN ct.type = 'credit' THEN ct.amount ELSE 0 END), 0) - 
             COALESCE(SUM(CASE WHEN ct.type = 'payment' THEN ct.amount ELSE 0 END), 0)) as total_balance
        FROM customers c
        LEFT JOIN credit_transactions ct ON c.id = ct.customer_id
        WHERE c.credit_limit > 0
        AND EXISTS (
            SELECT 1 
            FROM credit_transactions ct2 
            WHERE ct2.customer_id = c.id 
            AND ct2.type = 'credit'
        )
        GROUP BY c.id, c.name, c.email, c.phone, c.credit_limit
        HAVING (COALESCE(SUM(CASE WHEN ct.type = 'credit' THEN ct.amount ELSE 0 END), 0) - 
                COALESCE(SUM(CASE WHEN ct.type = 'payment' THEN ct.amount ELSE 0 END), 0)) > 0
        ORDER BY (COALESCE(SUM(CASE WHEN ct.type = 'credit' THEN ct.amount ELSE 0 END), 0) - 
                  COALESCE(SUM(CASE WHEN ct.type = 'payment' THEN ct.amount ELSE 0 END), 0)) DESC
    ");
    $stmt->execute();
    $overdueCreditors = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error fetching overdue creditors: " . $e->getMessage();
}

// Get low stock items
try {
    $stmt = $pdo->prepare("
        SELECT 
            p.*,
            c.name as category_name,
            CASE 
                WHEN p.quantity <= p.alert_quantity THEN 'low_stock'
                ELSE 'normal'
            END as stock_status
        FROM products p
        LEFT JOIN categories c ON p.category_id = c.id
        WHERE p.quantity <= p.alert_quantity
            AND p.status = 'active'
        ORDER BY p.quantity ASC
    ");
    $stmt->execute();
    $lowStockItems = $stmt->fetchAll();

    // Debug information
    error_log("Low stock items found: " . count($lowStockItems));
    foreach ($lowStockItems as $item) {
        error_log("Low stock item: " . $item['name'] . " - Quantity: " . $item['quantity'] . " - Alert Quantity: " . $item['alert_quantity']);
    }
} catch (PDOException $e) {
    $error = "Error fetching low stock items: " . $e->getMessage();
    error_log("Low stock query error: " . $e->getMessage());
}

// Get payment totals
try {
    $stmt = $pdo->prepare("
        SELECT 
            payment_method,
            COUNT(*) as total_transactions,
            SUM(total_amount) as total_amount,
            DATE(created_at) as payment_date
        FROM sales 
        WHERE payment_status = 'paid'
        GROUP BY payment_method, DATE(created_at)
        ORDER BY payment_date DESC, payment_method
    ");
    $stmt->execute();
    $paymentTotals = $stmt->fetchAll();

    // Calculate overall totals
    $overallTotals = [
        'cash' => 0,
        'mpesa' => 0,
        'credit' => 0,
        'total' => 0
    ];

    foreach ($paymentTotals as $payment) {
        $overallTotals[$payment['payment_method']] += $payment['total_amount'];
        $overallTotals['total'] += $payment['total_amount'];
    }
} catch (PDOException $e) {
    $error = "Error fetching payment totals: " . $e->getMessage();
}

// Get credit sales
try {
    $date_filter = isset($_GET['credit_date']) ? $_GET['credit_date'] : date('Y-m-d');
    
    // Debug: Log the date filter
    error_log("Credit Sales Date Filter: " . $date_filter);
    
    $stmt = $pdo->prepare("
        SELECT 
            s.*,
            c.name as customer_name,
            c.phone as customer_phone,
            c.email as customer_email,
            u.username as sales_person,
            DATEDIFF(CURRENT_DATE, s.created_at) as days_old,
            (s.total_amount - COALESCE(SUM(sp.amount), 0)) as remaining_balance,
            CASE 
                WHEN (s.total_amount - COALESCE(SUM(sp.amount), 0)) > 0 THEN 'pending'
                ELSE 'paid'
            END as payment_status
        FROM sales s
        LEFT JOIN customers c ON s.customer_id = c.id
        LEFT JOIN users u ON s.user_id = u.id
        LEFT JOIN sale_payments sp ON s.id = sp.sale_id
        WHERE s.payment_method = 'credit'
        AND DATE(s.created_at) = ?
        GROUP BY s.id, s.invoice_number, s.total_amount, s.created_at, s.user_id, 
                 c.name, c.phone, c.email, u.username
        HAVING payment_status IN ('pending', 'not paid')
        ORDER BY s.created_at DESC
    ");
    $stmt->execute([$date_filter]);
    $creditSales = $stmt->fetchAll();

    // Get total credit sales for the selected date
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_sales,
            COALESCE(SUM(total_amount), 0) as total_amount,
            COALESCE(SUM(
                total_amount - COALESCE((
                    SELECT SUM(amount) 
                    FROM sale_payments 
                    WHERE sale_id = s.id
                ), 0)
            ), 0) as total_remaining
        FROM sales s
        WHERE payment_method = 'credit'
        AND DATE(created_at) = ?
        AND EXISTS (
            SELECT 1 
            FROM sale_payments sp 
            WHERE sp.sale_id = s.id 
            GROUP BY sp.sale_id 
            HAVING (s.total_amount - COALESCE(SUM(sp.amount), 0)) > 0
        )
    ");
    $stmt->execute([$date_filter]);
    $creditTotals = $stmt->fetch();

} catch (PDOException $e) {
    $error = "Error fetching credit sales: " . $e->getMessage();
    error_log("Credit Sales Query Error: " . $e->getMessage());
}

// Get notifications with overdue credit alerts
$stmt = $pdo->prepare("
    SELECT 
        n.*,
        CASE 
            WHEN n.type = 'low_stock' THEN 'warning'
            WHEN n.type = 'expired' THEN 'info'
            WHEN n.type = 'overdue_credit' THEN 'danger'
            WHEN n.type = 'supplier_payment' THEN 'primary'
            WHEN n.type = 'inventory_alert' THEN 'warning'
            ELSE 'primary'
        END as badge_class,
        CASE 
            WHEN n.type = 'low_stock' THEN 'Low Stock Alert'
            WHEN n.type = 'expired' THEN 'Expired Product'
            WHEN n.type = 'overdue_credit' THEN 'Overdue Credit'
            WHEN n.type = 'supplier_payment' THEN 'Supplier Payment Due'
            WHEN n.type = 'inventory_alert' THEN 'Inventory Alert'
            ELSE 'General'
        END as notification_title,
        CASE 
            WHEN n.type = 'overdue_credit' THEN 1
            WHEN n.type = 'low_stock' THEN 2
            WHEN n.type = 'expired' THEN 3
            WHEN n.type = 'supplier_payment' THEN 4
            WHEN n.type = 'inventory_alert' THEN 5
            ELSE 6
        END as priority
    FROM notifications n
    WHERE n.is_read = 0
    AND (
        n.type != 'overdue_credit' 
        OR EXISTS (
            SELECT 1 
            FROM customers c
            LEFT JOIN credit_transactions ct ON c.id = ct.customer_id
            WHERE c.credit_limit > 0
            GROUP BY c.id
            HAVING (COALESCE(SUM(CASE WHEN ct.type = 'credit' THEN ct.amount ELSE 0 END), 0) - 
                    COALESCE(SUM(CASE WHEN ct.type = 'payment' THEN ct.amount ELSE 0 END), 0)) > 0
        )
    )
    ORDER BY 
        priority ASC,
        n.created_at DESC
");
$stmt->execute();
$notifications = $stmt->fetchAll();

// Get inventory alerts
try {
    $stmt = $pdo->prepare("
        SELECT 
            'inventory_alert' as type,
            CONCAT('Low Stock Alert: ', p.name) as title,
            CONCAT('Product ', p.name, ' is running low on stock. Current quantity: ', p.quantity) as message,
            NOW() as created_at,
            'warning' as badge_class,
            'Inventory Alert' as notification_title,
            5 as priority
        FROM products p
        WHERE p.quantity <= p.alert_quantity
        AND p.status = 'active'
        AND NOT EXISTS (
            SELECT 1 
            FROM notifications n 
            WHERE n.type = 'low_stock' 
            AND n.reference_id = p.id 
            AND n.created_at >= DATE_SUB(NOW(), INTERVAL 1 DAY)
        )
    ");
    $stmt->execute();
    $inventoryAlerts = $stmt->fetchAll();
    
    // Add inventory alerts to notifications
    foreach ($inventoryAlerts as $alert) {
        // Insert into notifications table
        $stmt = $pdo->prepare("
            INSERT INTO notifications (type, title, message, reference_id, created_at)
            VALUES (?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $alert['type'],
            $alert['title'],
            $alert['message'],
            $alert['reference_id']
        ]);
    }
    
    // Merge inventory alerts with notifications
    $notifications = array_merge($notifications, $inventoryAlerts);
    
    // Re-sort notifications
    usort($notifications, function($a, $b) {
        if ($a['priority'] === $b['priority']) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        }
        return $a['priority'] - $b['priority'];
    });
} catch (PDOException $e) {
    error_log("Error fetching inventory alerts: " . $e->getMessage());
}

// Get supplier payment notifications
try {
    $stmt = $pdo->prepare("
        SELECT 
            'supplier_payment' as type,
            CONCAT('Payment Due: ', s.name) as title,
            CONCAT('Payment of ', FORMAT(sr.total_amount, 2), ' is due to ', s.name) as message,
            sr.due_date as created_at,
            'primary' as badge_class,
            'Supplier Payment Due' as notification_title,
            4 as priority
        FROM supplier_receipts sr
        JOIN suppliers s ON sr.supplier_id = s.id
        WHERE sr.payment_status = 'pending'
        AND sr.due_date <= DATE_ADD(NOW(), INTERVAL 7 DAY)
        AND NOT EXISTS (
            SELECT 1 
            FROM notifications n 
            WHERE n.type = 'supplier_payment' 
            AND n.reference_id = sr.id 
            AND n.created_at >= DATE_SUB(NOW(), INTERVAL 1 DAY)
        )
    ");
    $stmt->execute();
    $supplierPayments = $stmt->fetchAll();
    
    // Add supplier payment notifications
    foreach ($supplierPayments as $payment) {
        // Insert into notifications table
        $stmt = $pdo->prepare("
            INSERT INTO notifications (type, title, message, reference_id, created_at)
            VALUES (?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $payment['type'],
            $payment['title'],
            $payment['message'],
            $payment['reference_id']
        ]);
    }
    
    // Merge supplier payment notifications
    $notifications = array_merge($notifications, $supplierPayments);
    
    // Final sort of all notifications
    usort($notifications, function($a, $b) {
        if ($a['priority'] === $b['priority']) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        }
        return $a['priority'] - $b['priority'];
    });
} catch (PDOException $e) {
    error_log("Error fetching supplier payment notifications: " . $e->getMessage());
}

// Mark notifications as read
if (!empty($notifications)) {
    try {
        // Start transaction
        $pdo->beginTransaction();
        
        // Mark notifications as read
        $stmt = $pdo->prepare("UPDATE notifications SET is_read = 1 WHERE is_read = 0");
        $stmt->execute();
        
        // Mark overdue credits as read by updating their status
        $stmt = $pdo->prepare("
            UPDATE credit_transactions ct
            JOIN customers c ON ct.customer_id = c.id
            SET ct.status = 'read'
            WHERE ct.type = 'credit' 
            AND ct.status = 'pending'
            AND EXISTS (
                SELECT 1 
                FROM (
                    SELECT customer_id,
                           SUM(CASE WHEN type = 'credit' THEN amount ELSE 0 END) - 
                           SUM(CASE WHEN type = 'payment' THEN amount ELSE 0 END) as balance
                    FROM credit_transactions
                    GROUP BY customer_id
                    HAVING balance > 0
                ) as unpaid
                WHERE unpaid.customer_id = ct.customer_id
            )
        ");
        $stmt->execute();
        
        // Commit transaction
        $pdo->commit();
    } catch (PDOException $e) {
        // Rollback transaction on error
        $pdo->rollBack();
        $error = "Error marking notifications as read: " . $e->getMessage();
    }
}

// Handle reminder actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && isset($_POST['customer_id'])) {
        $customerId = $_POST['customer_id'];
        $action = $_POST['action'];
        
        try {
            // Get customer details
            $stmt = $pdo->prepare("SELECT * FROM customers WHERE id = ?");
            $stmt->execute([$customerId]);
            $customer = $stmt->fetch();

            if ($action === 'send_email' && !empty($customer['email'])) {
                // Send email reminder
                $subject = "Payment Reminder - Overdue Credit";
                $message = "Dear " . $customer['name'] . ",\n\n";
                $message .= "This is a reminder that you have an overdue credit payment.\n";
                $message .= "Please settle your payment at your earliest convenience.\n\n";
                $message .= "Best regards,\nYour Store Name";

                // Use email API settings
                $headers = "From: " . $apiSettings['smtp_username'] . "\r\n";
                $headers .= "Reply-To: " . $apiSettings['smtp_username'] . "\r\n";
                $headers .= "X-Mailer: PHP/" . phpversion();

                mail($customer['email'], $subject, $message, $headers);
                $success = "Email reminder sent successfully";
            } 
            elseif ($action === 'send_sms' && !empty($customer['phone'])) {
                // Send SMS reminder
                $message = "Dear " . $customer['name'] . ", this is a reminder that you have an overdue credit payment. Please settle your payment at your earliest convenience.";
                
                // Use SMS API settings
                $apiKey = $apiSettings['sms_api_key'];
                $senderId = $apiSettings['sms_sender_id'];
                
                // Make API call to SMS provider
                $url = "https://api.smsprovider.com/send";
                $data = [
                    'api_key' => $apiKey,
                    'sender_id' => $senderId,
                    'phone' => $customer['phone'],
                    'message' => $message
                ];

                $ch = curl_init($url);
                curl_setopt($ch, CURLOPT_POST, 1);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                $response = curl_exec($ch);
                curl_close($ch);

                $success = "SMS reminder sent successfully";
            }
        } catch (Exception $e) {
            $error = "Error sending reminder: " . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Notifications - POS System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary-color: #2563eb;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --info-color: #3b82f6;
            --high-priority: #ef4444;
            --medium-priority: #f59e0b;
            --low-priority: #3b82f6;
        }

        body {
            font-family: 'Inter', sans-serif;
            background: #f8fafc;
            color: #1e293b;
        }

        .notification-card {
            background: white;
            border-radius: 12px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
            border: 1px solid #e2e8f0;
            margin-bottom: 1rem;
            overflow: hidden;
            position: relative;
        }

        .notification-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }

        .notification-card::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            bottom: 0;
            width: 4px;
            background: var(--primary-color);
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .notification-card:hover::before {
            opacity: 1;
        }

        .notification-header {
            padding: 1rem;
            border-bottom: 1px solid #e2e8f0;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .notification-icon {
            width: 40px;
            height: 40px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            color: white;
        }

        .notification-icon.warning { background: var(--warning-color); }
        .notification-icon.danger { background: var(--danger-color); }
        .notification-icon.success { background: var(--success-color); }
        .notification-icon.info { background: var(--info-color); }

        .notification-title {
            font-weight: 600;
            font-size: 1rem;
            margin: 0;
            color: #1e293b;
        }

        .notification-time {
            font-size: 0.875rem;
            color: #64748b;
            margin-top: 0.25rem;
        }

        .notification-body {
            padding: 1rem;
            color: #475569;
            font-size: 0.9375rem;
            line-height: 1.5;
        }

        .notification-actions {
            padding: 0.75rem 1rem;
            background: #f8fafc;
            border-top: 1px solid #e2e8f0;
            display: flex;
            justify-content: flex-end;
            gap: 0.5rem;
        }

        .btn-action {
            padding: 0.5rem 1rem;
            border-radius: 6px;
            font-size: 0.875rem;
            font-weight: 500;
            transition: all 0.2s ease;
        }

        .btn-action:hover {
            transform: translateY(-1px);
        }

        .empty-state {
            text-align: center;
            padding: 3rem 1rem;
            color: #64748b;
        }

        .empty-state i {
            font-size: 3rem;
            color: #cbd5e1;
            margin-bottom: 1rem;
        }

        .empty-state h3 {
            font-size: 1.25rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
            color: #475569;
        }

        .empty-state p {
            font-size: 0.9375rem;
            max-width: 400px;
            margin: 0 auto;
        }

        .notification-badge {
            position: absolute;
            top: -5px;
            right: -5px;
            background: var(--danger-color);
            color: white;
            border-radius: 50%;
            width: 20px;
            height: 20px;
            font-size: 0.75rem;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .priority-indicator {
            width: 8px;
            height: 8px;
            border-radius: 50%;
            margin-right: 8px;
        }

        .priority-high { background: var(--high-priority); }
        .priority-medium { background: var(--medium-priority); }
        .priority-low { background: var(--low-priority); }

        .date-group {
            margin-bottom: 2rem;
        }

        .date-header {
            font-size: 0.875rem;
            font-weight: 600;
            color: #64748b;
            margin-bottom: 1rem;
            padding: 0.5rem 0;
            border-bottom: 1px solid #e2e8f0;
        }

        .search-box {
            position: relative;
            margin-bottom: 1.5rem;
        }

        .search-box input {
            padding-left: 2.5rem;
            border-radius: 8px;
            border: 1px solid #e2e8f0;
            background: white;
        }

        .search-box i {
            position: absolute;
            left: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: #64748b;
        }

        .settings-panel {
            position: fixed;
            right: -300px;
            top: 0;
            bottom: 0;
            width: 300px;
            background: white;
            box-shadow: -2px 0 5px rgba(0,0,0,0.1);
            transition: right 0.3s ease;
            z-index: 1000;
            padding: 1.5rem;
        }

        .settings-panel.active {
            right: 0;
        }

        .settings-toggle {
            position: fixed;
            right: 1rem;
            bottom: 1rem;
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: var(--primary-color);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            box-shadow: 0 2px 5px rgba(0,0,0,0.2);
            z-index: 1001;
        }

        .load-more {
            text-align: center;
            margin-top: 2rem;
        }

        .load-more button {
            padding: 0.75rem 2rem;
            border-radius: 8px;
            background: white;
            border: 1px solid #e2e8f0;
            color: var(--primary-color);
            font-weight: 500;
            transition: all 0.2s ease;
        }

        .load-more button:hover {
            background: var(--primary-color);
            color: white;
        }

        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .notification-card {
            animation: slideIn 0.3s ease-out forwards;
        }

        .notification-card:nth-child(1) { animation-delay: 0.1s; }
        .notification-card:nth-child(2) { animation-delay: 0.2s; }
        .notification-card:nth-child(3) { animation-delay: 0.3s; }
        .notification-card:nth-child(4) { animation-delay: 0.4s; }
        .notification-card:nth-child(5) { animation-delay: 0.5s; }

        .page-header {
            background: white;
            padding: 1.5rem;
            border-radius: 12px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
            margin-bottom: 2rem;
        }

        .page-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: #1e293b;
            margin: 0;
        }

        .page-subtitle {
            color: #64748b;
            margin: 0.5rem 0 0;
            font-size: 0.9375rem;
        }

        .filter-buttons {
            display: flex;
            gap: 0.5rem;
            margin-bottom: 1.5rem;
            flex-wrap: wrap;
        }

        .filter-btn {
            padding: 0.5rem 1rem;
            border-radius: 6px;
            font-size: 0.875rem;
            font-weight: 500;
            background: white;
            border: 1px solid #e2e8f0;
            color: #64748b;
            transition: all 0.2s ease;
        }

        .filter-btn:hover, .filter-btn.active {
            background: var(--primary-color);
            border-color: var(--primary-color);
            color: white;
        }

        @media (max-width: 768px) {
            .notification-card {
                margin: 0 -1rem 1rem;
                border-radius: 0;
                border-left: none;
                border-right: none;
            }

            .page-header {
                margin: -1rem -1rem 1rem;
                border-radius: 0;
            }

            .settings-panel {
                width: 100%;
                right: -100%;
            }
        }

        .avatar-circle {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            font-size: 0.875rem;
        }

        .dropdown-item-form {
            margin: 0;
            padding: 0;
        }

        .dropdown-item-form button {
            width: 100%;
            text-align: left;
            padding: 0.5rem 1rem;
            border: none;
            background: none;
            color: inherit;
        }

        .dropdown-item-form button:hover {
            background-color: #f8f9fa;
        }

        .table > :not(caption) > * > * {
            padding: 1rem;
        }

        .table tbody tr:hover {
            background-color: #f8f9fa;
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
            <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>

                <div class="page-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="page-title">Notifications</h1>
                            <p class="page-subtitle">Stay updated with your system alerts and messages</p>
                        </div>
                        <div class="position-relative">
                            <button class="btn btn-primary" onclick="markAllRead()">
                                <i class="fas fa-check-double me-2"></i>Mark All as Read
                            </button>
                        </div>
                    </div>
                </div>

                <div class="filter-buttons">
                    <button class="filter-btn active" data-filter="overdue">
                        <i class="fas fa-exclamation-circle me-2"></i>Overdue Credit Alerts
                        <?php if (!empty($overdueCreditors)): ?>
                            <span class="badge bg-danger ms-2"><?php echo count($overdueCreditors); ?></span>
                        <?php endif; ?>
                    </button>
                    <button class="filter-btn" data-filter="low_stock">
                        <i class="fas fa-exclamation-triangle me-2"></i>Low Stock
                        <?php if (!empty($lowStockItems)): ?>
                            <span class="badge bg-warning ms-2"><?php echo count($lowStockItems); ?></span>
                        <?php endif; ?>
                    </button>
                    <button class="filter-btn" data-filter="credit_sales">
                        <i class="fas fa-credit-card me-2"></i>Today's Credit Sales
                        <?php if (!empty($creditSales)): ?>
                            <span class="badge bg-danger ms-2"><?php echo count($creditSales); ?></span>
                        <?php endif; ?>
                    </button>
                </div>

                <!-- Overdue Credit Alerts Section -->
                <?php if (!empty($overdueCreditors)): ?>
                <div id="overdue-section" class="card mb-4">
                    <div class="card-header bg-danger text-white">
                        <h3 class="card-title mb-0">
                            <i class="fas fa-exclamation-circle me-2"></i>Overdue Credit Alerts
                        </h3>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th>Customer</th>
                                        <th>Days Overdue</th>
                                        <th>Total Balance</th>
                                        <th>Credit Limit</th>
                                        <th>Last Payment</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($overdueCreditors as $creditor): ?>
                                        <tr>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="avatar-circle bg-danger text-white me-2">
                                                        <?php echo strtoupper(substr($creditor['name'], 0, 1)); ?>
                                                    </div>
                                                    <div>
                                                        <div class="fw-bold"><?php echo htmlspecialchars($creditor['name']); ?></div>
                                                        <small class="text-muted">
                                                            <?php echo htmlspecialchars($creditor['email']); ?>
                                                        </small>
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <span class="badge bg-danger">
                                                    <?php echo $creditor['days_overdue']; ?> days
                                                </span>
                                            </td>
                                            <td class="text-danger fw-bold">
                                                KES <?php echo number_format($creditor['total_balance'], 2); ?>
                                            </td>
                                            <td>
                                                KES <?php echo number_format($creditor['credit_limit'], 2); ?>
                                            </td>
                                            <td>
                                                <?php echo date('M d, Y', strtotime($creditor['last_payment_date'])); ?>
                                            </td>
                                            <td>
                                                <div class="btn-group">
                                                    <button type="button" class="btn btn-sm btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown">
                                                        <i class="fas fa-bell me-1"></i>Send Reminder
                                                    </button>
                                                    <ul class="dropdown-menu">
                                                        <?php if (!empty($creditor['email'])): ?>
                                                        <li>
                                                            <form method="post" class="dropdown-item-form">
                                                                <input type="hidden" name="customer_id" value="<?php echo $creditor['id']; ?>">
                                                                <input type="hidden" name="action" value="send_email">
                                                                <button type="submit" class="dropdown-item">
                                                                    <i class="fas fa-envelope me-2"></i>Send Email
                                                                </button>
                                                            </form>
                                                        </li>
                                                        <?php endif; ?>
                                                        <?php if (!empty($creditor['phone'])): ?>
                                                        <li>
                                                            <form method="post" class="dropdown-item-form">
                                                                <input type="hidden" name="customer_id" value="<?php echo $creditor['id']; ?>">
                                                                <input type="hidden" name="action" value="send_sms">
                                                                <button type="submit" class="dropdown-item">
                                                                    <i class="fas fa-sms me-2"></i>Send SMS
                                                                </button>
                                                            </form>
                                                        </li>
                                                        <?php endif; ?>
                                                    </ul>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                <?php else: ?>
                <div id="overdue-section" class="card mb-4">
                    <div class="card-header bg-success text-white">
                        <h3 class="card-title mb-0">
                            <i class="fas fa-check-circle me-2"></i>No Overdue Credits
                        </h3>
                    </div>
                    <div class="card-body">
                        <div class="text-center py-4">
                            <i class="fas fa-check-circle fa-3x text-success mb-3"></i>
                            <h4>All Credits are Up to Date</h4>
                            <p class="text-muted">There are no overdue credit payments at this time.</p>
                        </div>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Credit Sales Section -->
                <div id="credit-sales-section" style="display: none;">
                    <div class="card mb-4">
                        <div class="card-header bg-danger text-white">
                    <div class="d-flex justify-content-between align-items-center">
                                <h3 class="card-title mb-0">
                                    <i class="fas fa-credit-card me-2"></i>Today's Credit Sales
                                    <small class="ms-2">(<?php echo count($creditSales); ?> sales)</small>
                                </h3>
                                <form class="d-flex align-items-center" method="GET">
                                    <input type="hidden" name="filter" value="credit_sales">
                                    <input type="date" name="credit_date" class="form-control me-2" 
                                           value="<?php echo $date_filter; ?>" 
                                           onchange="this.form.submit()">
                                </form>
                        </div>
                        </div>
                        <div class="card-body">
                            <div class="row mb-4">
                                <div class="col-md-4">
                                    <div class="card bg-light">
                                        <div class="card-body text-center">
                                            <h5 class="card-title">Total Credit Sales</h5>
                                            <h3 class="text-danger">KES <?php echo number_format($creditTotals['total_amount'], 2); ?></h3>
                                            <small class="text-muted"><?php echo $creditTotals['total_sales']; ?> sales</small>
                    </div>
                </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card bg-light">
                                        <div class="card-body text-center">
                                            <h5 class="card-title">Remaining Balance</h5>
                                            <h3 class="text-danger">KES <?php echo number_format($creditTotals['total_remaining'], 2); ?></h3>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card bg-light">
                                        <div class="card-body text-center">
                                            <h5 class="card-title">Date</h5>
                                            <h3 class="text-danger"><?php echo date('M d, Y', strtotime($date_filter)); ?></h3>
                                        </div>
                                    </div>
                                </div>
                </div>

                            <?php if (!empty($creditSales)): ?>
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Invoice #</th>
                                                <th>Customer</th>
                                                <th>Amount</th>
                                                <th>Remaining</th>
                                                <th>Sales Person</th>
                                                <th>Time</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($creditSales as $sale): ?>
                                                <tr>
                                                    <td>
                                                        <a href="view_sale.php?id=<?php echo $sale['id']; ?>" class="text-primary">
                                                            <?php echo $sale['invoice_number']; ?>
                                                        </a>
                                                    </td>
                                                    <td>
                                                        <div class="d-flex align-items-center">
                                                            <div class="avatar-circle bg-danger text-white me-2">
                                                                <?php echo strtoupper(substr($sale['customer_name'], 0, 1)); ?>
                                                            </div>
                                                            <div>
                                                                <div class="fw-bold"><?php echo htmlspecialchars($sale['customer_name']); ?></div>
                                                                <small class="text-muted">
                                                                    <?php echo htmlspecialchars($sale['customer_phone']); ?>
                                                                </small>
                                                            </div>
                                                        </div>
                                                    </td>
                                                    <td class="text-danger">KES <?php echo number_format($sale['total_amount'], 2); ?></td>
                                                    <td class="text-danger">KES <?php echo number_format($sale['remaining_balance'], 2); ?></td>
                                                    <td><?php echo htmlspecialchars($sale['sales_person']); ?></td>
                                                    <td><?php echo date('g:i A', strtotime($sale['created_at'])); ?></td>
                                                    <td>
                                                        <div class="btn-group">
                                                            <a href="view_sale.php?id=<?php echo $sale['id']; ?>" class="btn btn-sm btn-primary">
                                                                <i class="fas fa-eye"></i>
                                                            </a>
                                                            <?php if ($sale['remaining_balance'] > 0): ?>
                                                            <a href="record_payment.php?sale_id=<?php echo $sale['id']; ?>" class="btn btn-sm btn-success">
                                                                <i class="fas fa-money-bill"></i>
                                                            </a>
                                                            <?php endif; ?>
                                                        </div>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            <?php else: ?>
                                <div class="alert alert-info">
                                    <i class="fas fa-info-circle me-2"></i>
                                    No credit sales found for <?php echo date('M d, Y', strtotime($date_filter)); ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <div id="notifications-section" style="display: none;">
                    <?php if (empty($notifications) && empty($lowStockItems) && empty($creditSales)): ?>
                    <div class="empty-state">
                        <i class="fas fa-bell-slash"></i>
                        <h3>No Notifications</h3>
                        <p>You're all caught up! There are no new notifications at this time.</p>
                    </div>
                <?php else: ?>
                    <?php 
                    $notificationCount = 0;
                    $categoryCounts = [
                        'low_stock' => count($lowStockItems),
                        'credit_sales' => count($creditSales)
                    ];
                    
                    // Display low stock items
                    if (!empty($lowStockItems)): 
                    ?>
                        <div class="notification-card" 
                             data-type="warning"
                             data-category="low_stock">
                            <div class="notification-header">
                                <div class="notification-icon warning">
                                    <i class="fas fa-exclamation-triangle"></i>
                                </div>
                                <div class="flex-grow-1">
                                    <h3 class="notification-title">
                                        Low Stock Alert: <?php echo htmlspecialchars($lowStockItems[0]['name']); ?>
                                    </h3>
                                    <div class="notification-time">
                                        <i class="far fa-clock me-1"></i>
                                        Last Updated: <?php echo date('M d, Y g:i A', strtotime($lowStockItems[0]['updated_at'])); ?>
                                    </div>
                                </div>
                                <span class="badge bg-warning">Low Stock</span>
                            </div>
                            <div class="notification-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <p class="mb-1">
                                            <strong>Current Stock:</strong> 
                                            <span class="text-danger"><?php echo $lowStockItems[0]['quantity']; ?> units</span>
                                        </p>
                                        <p class="mb-1">
                                            <strong>Alert Quantity:</strong> 
                                            <?php echo $lowStockItems[0]['alert_quantity']; ?> units
                                        </p>
                                    </div>
                                    <div class="col-md-6">
                                        <p class="mb-1">
                                            <strong>Category:</strong> 
                                            <?php echo htmlspecialchars($lowStockItems[0]['category_name']); ?>
                                        </p>
                                        <p class="mb-1">
                                            <strong>Serial No:</strong> 
                                            <?php echo htmlspecialchars($lowStockItems[0]['serial_no']); ?>
                                        </p>
                                    </div>
                                </div>
                            </div>
                            <div class="notification-actions">
                                <a href="../pages/products.php?product_id=<?php echo $lowStockItems[0]['id']; ?>" class="btn btn-sm btn-primary">
                                    <i class="fas fa-external-link-alt me-1"></i>View Product
                                </a>
                            </div>
                        </div>
                    <?php endif; ?>

                    <!-- Display other notifications -->
                    <?php foreach ($notifications as $notification): 
                        if ($notification['type'] !== 'low_stock' && $notification['type'] !== 'credit_sales') {
                            $notificationCount++;
                            if (isset($categoryCounts[$notification['type']])) {
                                $categoryCounts[$notification['type']]++;
                            }
                    ?>
                        <div class="notification-card" 
                             data-type="<?php echo $notification['badge_class']; ?>"
                             data-category="<?php echo $notification['type']; ?>">
                            <div class="notification-header">
                                <div class="notification-icon <?php echo $notification['badge_class']; ?>">
                                    <?php
                                    switch($notification['type']) {
                                        case 'low_stock':
                                            echo '<i class="fas fa-exclamation-triangle"></i>';
                                            break;
                                        case 'credit_sales':
                                            echo '<i class="fas fa-clock"></i>';
                                            break;
                                        case 'overdue_credit':
                                            echo '<i class="fas fa-exclamation-circle"></i>';
                                            break;
                                    }
                                    ?>
                                </div>
                                <div class="flex-grow-1">
                                    <h3 class="notification-title">
                                        <?php echo $notification['notification_title']; ?>
                                    </h3>
                                    <div class="notification-time">
                                        <i class="far fa-clock me-1"></i>
                                        <?php echo date('M d, Y g:i A', strtotime($notification['created_at'])); ?>
                                    </div>
                                </div>
                                <?php if ($notification['type'] === 'low_stock'): ?>
                                    <span class="badge bg-warning">Low Stock</span>
                                <?php elseif ($notification['type'] === 'credit_sales'): ?>
                                    <span class="badge bg-danger">Credit Sale</span>
                                <?php endif; ?>
                            </div>
                            <div class="notification-body">
                                <?php echo htmlspecialchars($notification['message']); ?>
                            </div>
                            <?php if ($notification['type'] === 'low_stock' || $notification['type'] === 'credit_sales'): ?>
                                <div class="notification-actions">
                                    <a href="inventory.php" class="btn btn-sm btn-primary">
                                        <i class="fas fa-external-link-alt me-1"></i>View Details
                                    </a>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php 
                        }
                    endforeach; 
                    ?>
                <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <div class="settings-toggle" onclick="toggleSettings()">
        <i class="fas fa-cog"></i>
    </div>

    <div class="settings-panel">
        <h3 class="mb-4">Notification Settings</h3>
        <div class="mb-3">
            <label class="form-label">Notification Sound</label>
            <div class="form-check form-switch">
                <input class="form-check-input" type="checkbox" id="soundToggle" checked>
                <label class="form-check-label" for="soundToggle">Enable sound</label>
            </div>
        </div>
        <div class="mb-3">
            <label class="form-label">Email Notifications</label>
            <div class="form-check form-switch">
                <input class="form-check-input" type="checkbox" id="emailToggle" checked>
                <label class="form-check-label" for="emailToggle">Enable email notifications</label>
            </div>
        </div>
        <div class="mb-3">
            <label class="form-label">Desktop Notifications</label>
            <div class="form-check form-switch">
                <input class="form-check-input" type="checkbox" id="desktopToggle" checked>
                <label class="form-check-label" for="desktopToggle">Enable desktop notifications</label>
            </div>
        </div>
        <hr>
        <div class="mb-3">
            <label class="form-label">Notification Types</label>
            <div class="form-check">
                <input class="form-check-input" type="checkbox" id="typeLowStock" checked>
                <label class="form-check-label" for="typeLowStock">Low Stock Alerts</label>
            </div>
            <div class="form-check">
                <input class="form-check-input" type="checkbox" id="typeExpired" checked>
                <label class="form-check-label" for="typeExpired">Expiration Notices</label>
            </div>
            <div class="form-check">
                <input class="form-check-input" type="checkbox" id="typePayment" checked>
                <label class="form-check-label" for="typePayment">Payment Updates</label>
            </div>
        </div>
    </div>

    <audio id="notificationSound" src="notification.mp3" preload="auto"></audio>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Update notification counts
        function updateNotificationCounts() {
            const counts = {
                'low_stock': document.querySelectorAll('.notification-card[data-category="low_stock"]').length,
                'credit_sales': document.querySelectorAll('.notification-card[data-category="credit_sales"]').length,
                'payments': document.querySelectorAll('.notification-card[data-category="payments"]').length
            };

            document.getElementById('low-stock-count').textContent = counts.low_stock;
            document.getElementById('credit-sales-count').textContent = counts.credit_sales;
            document.getElementById('payments-count').textContent = counts.payments;
        }

        // Filter functionality
        document.querySelectorAll('.filter-btn').forEach(button => {
            button.addEventListener('click', () => {
                // Remove active class from all buttons
                document.querySelectorAll('.filter-btn').forEach(btn => btn.classList.remove('active'));
                // Add active class to clicked button
                button.classList.add('active');

                const filter = button.dataset.filter;
                const overdueSection = document.getElementById('overdue-section');
                const notificationsSection = document.getElementById('notifications-section');
                const creditSalesSection = document.getElementById('credit-sales-section');
                const notificationCards = document.querySelectorAll('.notification-card');

                if (filter === 'overdue') {
                    // Show overdue credit alerts
                    overdueSection.style.display = 'block';
                    notificationsSection.style.display = 'none';
                    creditSalesSection.style.display = 'none';
                } else if (filter === 'credit_sales') {
                    // Show credit sales section
                    overdueSection.style.display = 'none';
                    notificationsSection.style.display = 'none';
                    creditSalesSection.style.display = 'block';
                } else if (filter === 'low_stock') {
                    // Show low stock notifications
                    overdueSection.style.display = 'none';
                    notificationsSection.style.display = 'block';
                    creditSalesSection.style.display = 'none';

                    // Filter notifications based on category
                    notificationCards.forEach(card => {
                        const category = card.dataset.category;
                        card.style.display = category === 'low_stock' ? 'block' : 'none';
                    });
                }
            });
        });

        // Initialize
        document.addEventListener('DOMContentLoaded', function() {
            // Show overdue credit alerts by default
            const overdueButton = document.querySelector('.filter-btn[data-filter="overdue"]');
            if (overdueButton) {
                overdueButton.click();
            }
            
            // Update notification counts
            updateNotificationCounts();
        });

        // Mark all as read
        function markAllRead() {
            // Create form data
            const formData = new FormData();
            formData.append('action', 'mark_all_read');

            // Send AJAX request
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(() => {
                // Remove notification badges
                document.querySelectorAll('.notification-badge').forEach(badge => badge.remove());
                
                // Update notification count in topbar
                const topbarNotificationCount = document.querySelector('#notificationDropdown .badge');
                if (topbarNotificationCount) {
                    topbarNotificationCount.textContent = '0';
                    topbarNotificationCount.style.display = 'none';
                }
                
                // Update all filter button badges
                const filterBadges = document.querySelectorAll('.filter-btn .badge');
                filterBadges.forEach(badge => {
                    badge.textContent = '0';
                    badge.style.display = 'none';
                });

                // Hide overdue credit alerts section
                const overdueSection = document.getElementById('overdue-section');
                if (overdueSection) {
                    overdueSection.style.display = 'none';
                }

                // Show "No Overdue Credits" message
                const noOverdueSection = document.createElement('div');
                noOverdueSection.id = 'overdue-section';
                noOverdueSection.className = 'card mb-4';
                noOverdueSection.innerHTML = `
                    <div class="card-header bg-success text-white">
                        <h3 class="card-title mb-0">
                            <i class="fas fa-check-circle me-2"></i>No Overdue Credits
                        </h3>
                    </div>
                    <div class="card-body">
                        <div class="text-center py-4">
                            <i class="fas fa-check-circle fa-3x text-success mb-3"></i>
                            <h4>All Credits are Up to Date</h4>
                            <p class="text-muted">There are no overdue credit payments at this time.</p>
                        </div>
                    </div>
                `;
                document.querySelector('.col-md-10').insertBefore(noOverdueSection, document.querySelector('.filter-buttons'));
                
                // Update notification count in localStorage
                localStorage.setItem('notificationCount', '0');
                
                // Force refresh the page to ensure all counts are updated
                setTimeout(() => {
                    window.location.reload();
                }, 500);
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error marking notifications as read. Please try again.');
            });
        }

        // Settings panel toggle
        function toggleSettings() {
            document.querySelector('.settings-panel').classList.toggle('active');
        }

        // Play notification sound
        function playNotificationSound() {
            const sound = document.getElementById('notificationSound');
            if (document.getElementById('soundToggle').checked) {
                sound.play();
            }
        }

        // Check for new notifications periodically
        setInterval(() => {
            fetch('check_new_notifications.php')
                .then(response => response.json())
                .then(data => {
                    if (data.hasNew) {
                        playNotificationSound();
                        // Refresh notifications
                        location.reload();
                    }
                })
                .catch(error => console.error('Error:', error));
        }, 30000); // Check every 30 seconds
    </script>
    <?php include '../includes/footer.php'; ?>
</body>
</html> 